/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.validation.qwac;

import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.model.http.ResponseEnvelope;
import eu.europa.esig.dss.model.x509.CertificateToken;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.AdvancedDataLoader;
import eu.europa.esig.dss.spi.client.http.NativeHTTPDataLoader;
import eu.europa.esig.dss.spi.client.http.Protocol;
import eu.europa.esig.dss.spi.signature.AdvancedSignature;
import eu.europa.esig.dss.spi.validation.CertificateVerifier;
import eu.europa.esig.dss.spi.validation.CertificateVerifierBuilder;
import eu.europa.esig.dss.spi.validation.ValidationContext;
import eu.europa.esig.dss.spi.x509.CommonCertificateSource;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.validation.AbstractCertificateValidator;
import eu.europa.esig.dss.validation.SignedDocumentValidator;
import eu.europa.esig.dss.validation.executor.certificate.CertificateProcessExecutor;
import eu.europa.esig.dss.validation.executor.certificate.qwac.QWACCertificateProcessExecutor;
import eu.europa.esig.dss.validation.qwac.QWACUtils;
import eu.europa.esig.dss.validation.reports.CertificateReports;
import eu.europa.esig.dss.validation.reports.diagnostic.DiagnosticDataBuilder;
import eu.europa.esig.dss.validation.reports.diagnostic.QWACCertificateDiagnosticDataBuilder;
import java.security.cert.Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QWACValidator
extends AbstractCertificateValidator<CertificateReports, CertificateProcessExecutor> {
    private static final Logger LOG = LoggerFactory.getLogger(QWACValidator.class);
    private static final String QWAC_VALIDATION_POLICY_LOCATION = "/policy/qwac-constraint.xml";
    private final String url;
    private CertificateToken certificateToken;
    private AdvancedDataLoader dataLoader;

    private QWACValidator(String url) {
        this(url, null);
    }

    private QWACValidator(String url, CertificateToken certificateToken) {
        Objects.requireNonNull(url, "URL cannot be null!");
        this.url = url;
        this.certificateToken = certificateToken;
    }

    protected AdvancedDataLoader getDataLoader() {
        if (this.dataLoader == null) {
            LOG.info("NativeHTTPDataLoader is instantiated for QWACValidator execution.");
            this.dataLoader = new NativeHTTPDataLoader();
        }
        return this.dataLoader;
    }

    public void setDataLoader(AdvancedDataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public static QWACValidator fromUrl(String url) {
        return new QWACValidator(url);
    }

    public static QWACValidator fromUrlAndCertificate(String url, CertificateToken certificateToken) {
        return new QWACValidator(url, certificateToken);
    }

    @Override
    protected DiagnosticDataBuilder prepareDiagnosticDataBuilder() {
        ResponseEnvelope response = this.connectToUrl();
        this.assertResponseValid(response);
        List<CertificateToken> tlsCertificates = this.toCertificateTokenList(response.getTLSCertificates());
        String tlsCertificateBindingUrl = this.readTLSCertificateBindingUrl(response);
        SignedDocumentValidator signedDocumentValidator = this.initSignedDocumentValidator(tlsCertificateBindingUrl, tlsCertificates);
        AdvancedSignature signature = this.getTLSCertificateBindingSignature(signedDocumentValidator);
        CertificateToken tlsCertificate = this.getTLSCertificate(tlsCertificates);
        ValidationContext validationContext = this.prepareValidationContext(tlsCertificate, tlsCertificates, signature);
        this.validateContext(validationContext);
        return this.createDiagnosticDataBuilder(validationContext, signedDocumentValidator, tlsCertificate, tlsCertificateBindingUrl, signature);
    }

    protected ResponseEnvelope connectToUrl() {
        String trimmedUrl = Utils.trim(this.url);
        if (Protocol.isHttpUrl(trimmedUrl)) {
            return this.getDataLoader().requestGet(this.url, true, false);
        }
        throw new UnsupportedOperationException(String.format("DSS framework supports only HTTP(S) certificate extraction. Obtained URL : '%s'", this.url));
    }

    protected ValidationContext prepareValidationContext(CertificateToken tlsCertificate, List<CertificateToken> otherTlsCertificates, AdvancedSignature signature) {
        CertificateVerifier certificateVerifierForValidation = this.getOfflineCertificateVerifier();
        ValidationContext validationContext = super.prepareValidationContext(certificateVerifierForValidation);
        validationContext.addCertificateTokenForVerification(tlsCertificate);
        CommonCertificateSource adjunctCertificateSource = new CommonCertificateSource();
        for (CertificateToken certificate : otherTlsCertificates) {
            adjunctCertificateSource.addCertificate(certificate);
        }
        certificateVerifierForValidation.addAdjunctCertSources(adjunctCertificateSource);
        if (signature != null) {
            validationContext.addSignatureForVerification(signature);
        }
        return validationContext;
    }

    private CertificateToken getTLSCertificate(List<CertificateToken> tlsCertificates) {
        if (this.certificateToken != null) {
            return this.certificateToken;
        }
        if (Utils.isCollectionNotEmpty(tlsCertificates)) {
            this.certificateToken = tlsCertificates.iterator().next();
            return this.certificateToken;
        }
        throw new DSSException("No valid TLS/SSL certificates have been obtained from the URL '{}'.");
    }

    private SignedDocumentValidator initSignedDocumentValidator(String tlsCertificateBindingUrl, List<CertificateToken> tlsCertificates) {
        byte[] tlsCertificateBindingSignatureBytes;
        if (tlsCertificateBindingUrl != null && (tlsCertificateBindingSignatureBytes = this.getDataLoader().get(tlsCertificateBindingUrl)) != null) {
            InMemoryDocument signatureDocument = new InMemoryDocument(tlsCertificateBindingSignatureBytes, tlsCertificateBindingUrl);
            SignedDocumentValidator documentValidator = SignedDocumentValidator.fromDocument(signatureDocument);
            documentValidator.setCertificateVerifier(this.getOfflineCertificateVerifier());
            documentValidator.setDetachedContents(this.toDetachedDocumentsList(tlsCertificates));
            return documentValidator;
        }
        return null;
    }

    private AdvancedSignature getTLSCertificateBindingSignature(SignedDocumentValidator signedDocumentValidator) {
        if (signedDocumentValidator != null) {
            List<AdvancedSignature> signatures = signedDocumentValidator.getSignatures();
            if (signatures.size() == 1) {
                return signatures.get(0);
            }
            LOG.warn("Only one signature is expected within the TLS Certificate Binding URL. Obtained : {}.", (Object)signatures.size());
        }
        return null;
    }

    private List<CertificateToken> toCertificateTokenList(Certificate[] certificates) {
        if (Utils.isArrayEmpty(certificates)) {
            return Collections.emptyList();
        }
        return Arrays.stream(certificates).map(this::toCertificateToken).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private CertificateToken toCertificateToken(Certificate certificate) {
        try {
            return DSSUtils.loadCertificate(certificate.getEncoded());
        }
        catch (Exception e) {
            LOG.warn("Unable to load certificate : {}. The entry is skipped.", (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    private List<DSSDocument> toDetachedDocumentsList(List<CertificateToken> certificates) {
        return certificates.stream().map(this::toDSSDocument).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private DSSDocument toDSSDocument(CertificateToken certificate) {
        try {
            return new InMemoryDocument(certificate.getEncoded(), this.identifierProvider.getIdAsString(certificate));
        }
        catch (Exception e) {
            LOG.warn("Unable to load certificate : {}. The entry is skipped.", (Object)e.getMessage(), (Object)e);
            return null;
        }
    }

    protected String readTLSCertificateBindingUrl(ResponseEnvelope responseEnvelope) {
        return QWACUtils.getTLSCertificateBindingUrl(responseEnvelope.getHeaders());
    }

    protected void assertResponseValid(ResponseEnvelope response) {
        if (this.certificateToken == null && Utils.isArrayEmpty(response.getTLSCertificates())) {
            throw new IllegalArgumentException(String.format("No TSL/SSL certificates have been returned from the URL '%s'. Please ensure the URL is valid and uses the HTTP(S) scheme, or provide the TLS/SSL certificate on validation explicitly.", this.url));
        }
    }

    protected DiagnosticDataBuilder createDiagnosticDataBuilder(ValidationContext validationContext, SignedDocumentValidator signedDocumentValidator, CertificateToken tlsCertificate, String tlsCertificateBindingUrl, AdvancedSignature signature) {
        QWACCertificateDiagnosticDataBuilder diagnosticDataBuilder = (QWACCertificateDiagnosticDataBuilder)super.createDiagnosticDataBuilder(validationContext);
        diagnosticDataBuilder = diagnosticDataBuilder.websiteUrl(this.url).tlsCertificate(tlsCertificate).tlsCertificateBindingUrl(tlsCertificateBindingUrl).tlsCertificateBindingSignature(signature);
        if (signedDocumentValidator != null) {
            diagnosticDataBuilder = diagnosticDataBuilder.setSignatureDiagnosticDataBuilder(signedDocumentValidator.initializeDiagnosticDataBuilder());
        }
        return diagnosticDataBuilder.foundSignatures(validationContext.getProcessedSignatures()).documentCertificateSource(validationContext.getDocumentCertificateSource());
    }

    protected CertificateVerifier getOfflineCertificateVerifier() {
        return new CertificateVerifierBuilder(this.certificateVerifier).buildCompleteCopyForValidation();
    }

    @Override
    protected DiagnosticDataBuilder initDiagnosticDataBuilder() {
        return new QWACCertificateDiagnosticDataBuilder();
    }

    @Override
    protected String getDefaultValidationPolicyPath() {
        return QWAC_VALIDATION_POLICY_LOCATION;
    }

    @Override
    protected CertificateProcessExecutor provideProcessExecutorInstance() {
        if (this.processExecutor == null) {
            this.processExecutor = this.getDefaultProcessExecutor();
        }
        if (this.certificateToken != null) {
            ((CertificateProcessExecutor)this.processExecutor).setCertificateId(this.identifierProvider.getIdAsString(this.certificateToken));
        }
        return (CertificateProcessExecutor)this.processExecutor;
    }

    @Override
    public CertificateProcessExecutor getDefaultProcessExecutor() {
        return new QWACCertificateProcessExecutor();
    }
}

